function formatBytes(bytes) {
  if (bytes < 1024) return `${bytes} B`;
  if (bytes < 1024 * 1024) return `${(bytes / 1024).toFixed(1)} KB`;
  return `${(bytes / (1024 * 1024)).toFixed(1)} MB`;
}

let lastImageUrls = [];

document.addEventListener('DOMContentLoaded', () => {
  chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
    chrome.tabs.sendMessage(
      tabs[0].id,
      { action: 'getImages' },
      (imageUrls) => {
        const container = document.getElementById('image-list');
        container.innerHTML = '';

        if (!imageUrls || imageUrls.length === 0) {
          container.textContent = 'No image found';
          return;
        }

        lastImageUrls = imageUrls; // 一括用に保存

        imageUrls.forEach(url => {
          const wrapper = document.createElement('div');
          wrapper.className = 'thumb';

          const img = document.createElement('img');
          img.src = url;
          img.loading = 'lazy';
          img.title = url;

          const info = document.createElement('div');
          info.style.fontSize = '12px';
          info.textContent = 'Loading...';

          img.onload = async () => {
            const width = img.naturalWidth;
            const height = img.naturalHeight;

            try {
              const res = await fetch(url);
              const blob = await res.blob();
              const size = formatBytes(blob.size);
              info.textContent = `${width}x${height}px\n${size}`;
            } catch {
              info.textContent = `${width}x${height}px`;
            }
          };

          wrapper.appendChild(img);
          wrapper.appendChild(info);

          wrapper.addEventListener('click', () => {
            let finalUrl = img.src;
            const urlObj = new URL(finalUrl);
            let filename = urlObj.pathname.split('/').pop().split('?')[0];

            // === Twitter
            if (urlObj.hostname.includes('pbs.twimg.com')) {
              const id = urlObj.pathname.split('/').pop();
              const format = urlObj.searchParams.get('format') || 'jpg';
              finalUrl = `https://pbs.twimg.com/media/${id}?format=${format}&name=orig`;
              filename = `${id}.${format}`;
            }

            // === Pixiv
            else if (urlObj.hostname.includes('pximg.net') && finalUrl.includes('/img-master/')) {
              finalUrl = finalUrl.replace('/img-master/', '/img-original/')
                                 .replace(/_master\d+\.(jpg|png|webp)/, '.$1');
              filename = finalUrl.split('/').pop().split('?')[0];
            }

            const a = document.createElement('a');
            a.href = finalUrl;
            a.download = filename;
            a.click();
          });

          container.appendChild(wrapper);
        });
      }
    );
  });

  // URL一括コピー
  document.getElementById('copy-urls')?.addEventListener('click', () => {
    const text = lastImageUrls.join('\n');
    navigator.clipboard.writeText(text).then(() => {
      alert('Link Copied');
    }).catch(err => {
      alert('Link Copy Failed: ' + err);
    });
  });

  // 一括ダウンロード
  document.getElementById('download-all')?.addEventListener('click', () => {
    lastImageUrls.forEach((src) => {
      let finalUrl = src;
      const urlObj = new URL(finalUrl);
      let filename = urlObj.pathname.split('/').pop().split('?')[0];

      // Twitter
      if (urlObj.hostname.includes('pbs.twimg.com')) {
        const id = urlObj.pathname.split('/').pop();
        const format = urlObj.searchParams.get('format') || 'jpg';
        finalUrl = `https://pbs.twimg.com/media/${id}?format=${format}&name=orig`;
        filename = `${id}.${format}`;
      }

      // Pixiv
      else if (urlObj.hostname.includes('pximg.net') && finalUrl.includes('/img-master/')) {
        finalUrl = finalUrl.replace('/img-master/', '/img-original/')
                           .replace(/_master\d+\.(jpg|png|webp)/, '.$1');
        filename = finalUrl.split('/').pop().split('?')[0];
      }

      const a = document.createElement('a');
      a.href = finalUrl;
      a.download = filename;
      a.click();
    });
  });
});
