// ---- Shadow DOMでツールチップ作成 ----
const host = document.createElement("div");
host.id = "link-hover-tooltip-host";
document.body.appendChild(host);

const shadow = host.attachShadow({ mode: "open" });
shadow.innerHTML = `
  <style>
    #tooltip {
      position: absolute;
      z-index: 99999;
      background-color: rgba(0, 0, 0, 0.9);
      color: white;
      padding: 10px 14px;
      font-size: 14px;
      max-width: 600px;
      word-break: break-word;
      border-radius: 6px;
      pointer-events: none;
      box-shadow: 0 0 10px rgba(0, 0, 0, 0.4);
      transition: opacity 0.2s ease;
      font-family: sans-serif;
    }
    .ad-label {
      background: #f00;
      color: white;
      padding: 2px 4px;
      margin-left: 6px;
      font-size: 12px;
      border-radius: 3px;
    }
    .decoded {
      color: #ccc;
    }
    .realurl {
      color: #8f8;
    }
  </style>
  <div id="tooltip" style="display: none;"></div>
`;

const tooltip = shadow.querySelector("#tooltip");

// ---- 広告ドメイン判定 ----
const adDomains = [
  "googleads.g.doubleclick.net",
  "tpc.googlesyndication.com",
  "adclick.g.doubleclick.net",
  "pagead2.googlesyndication.com",
  "a.impactradius-go.com",
  "affiliate.rakuten.co.jp",
  "af.moshimo.com",
  "click.linksynergy.com"
];

function decodeURLSafely(url) {
  try {
    return decodeURI(url);
  } catch (e) {
    return url;
  }
}

function extractRealURL(adUrl) {
  try {
    const urlObj = new URL(adUrl);
    const redirectParam = urlObj.searchParams.get("redirect") || urlObj.searchParams.get("target");
    if (redirectParam) {
      return decodeURIComponent(redirectParam);
    }
  } catch (e) {}
  return null;
}

function isAdLink(url) {
  try {
    const parsed = new URL(url);
    return adDomains.some(ad => parsed.hostname.endsWith(ad));
  } catch (e) {
    return false;
  }
}

function showTooltip(content, x, y) {
  tooltip.innerHTML = content;
  tooltip.style.left = `${x + 15}px`;
  tooltip.style.top = `${y + 15}px`;
  tooltip.style.display = "block";
}

// ---- 通常リンクの表示 ----
function showLinkTooltip(url, x, y) {
  const decoded = decodeURLSafely(url);
  const realUrl = extractRealURL(url);
  const isAd = isAdLink(url);

  const content = `
    <div>
      <strong style="color: ${isAd ? '#f88' : 'white'};">
        ${url}
        ${isAd ? '<span class="ad-label">広告リンク</span>' : ''}
      </strong>
    </div>
    ${url !== decoded ? `<div class="decoded">${decoded}</div>` : ""}
    ${realUrl ? `<div class="realurl">→ ${realUrl}</div>` : ""}
  `;
  showTooltip(content, x, y);
}

document.addEventListener("mouseover", (e) => {
  const linkEl = e.target.closest("a");
  if (linkEl && linkEl.href) {
    showLinkTooltip(linkEl.href, e.pageX, e.pageY);
  }
});

document.addEventListener("mouseout", (e) => {
  if (e.target.closest("a")) {
    tooltip.style.display = "none";
  }
});

document.addEventListener("mousemove", (e) => {
  const linkEl = e.target.closest("a");
  if (linkEl && linkEl.href) {
    tooltip.style.left = `${e.pageX + 15}px`;
    tooltip.style.top = `${e.pageY + 15}px`;
  }
});

// ---- iframe広告の検出 ----
function addIframeListeners() {
  document.querySelectorAll("iframe").forEach((iframe) => {
    if (iframe.dataset.hoverTooltip) return; // 重複防止
    iframe.dataset.hoverTooltip = "true";

    iframe.addEventListener("mouseenter", (e) => {
      const src = iframe.src || "(外部広告: URL取得不可)";
      const isAdIframe = adDomains.some(ad => src.includes(ad));
      const content = `
        <div>
          <strong style="color: ${isAdIframe ? '#f88' : 'white'};">
            ${src || "iframe広告"}
            ${isAdIframe ? '<span class="ad-label">外部広告</span>' : '(iframe)'}
          </strong>
        </div>
      `;
      showTooltip(content, e.pageX, e.pageY);
    });

    iframe.addEventListener("mouseleave", () => {
      tooltip.style.display = "none";
    });

    iframe.addEventListener("mousemove", (e) => {
      tooltip.style.left = `${e.pageX + 15}px`;
      tooltip.style.top = `${e.pageY + 15}px`;
    });
  });
}

// ページロード後にiframe監視
addIframeListeners();
const observer = new MutationObserver(addIframeListeners);
observer.observe(document.body, { childList: true, subtree: true });

// すでに存在している observer に以下を追加
function isVisible(el) {
  const style = window.getComputedStyle(el);
  return style && style.visibility !== 'hidden' && style.display !== 'none' && parseFloat(style.opacity) > 0;
}

function findNearestLink(target) {
  // 通常の aタグだけでなく、親にリンクっぽい要素があるかも調べる
  let el = target;
  while (el && el !== document.body) {
    if (el.tagName === 'A' && el.href && isVisible(el)) {
      return el;
    }
    el = el.parentElement;
  }
  return null;
}
