let currentBookmarks = [];

function loadFolders() {
  chrome.bookmarks.getTree((tree) => {
    const folderList = document.getElementById("folder-list");
    folderList.innerHTML = "";

    function render(node, depth = 0) {
      if (node.children) {
        // (no name)（＝title が空）で、最上位階層（depth === 0）の場合はスキップ
        if (!node.title && depth === 0) return;

        const li = document.createElement("li");
        li.style.paddingLeft = `${depth * 16}px`; // ◀ インデントを階層に応じて

        const icon = document.createElement("span");
        icon.textContent = "📁";
        icon.style.marginRight = "4px";
        li.appendChild(icon);

        const label = document.createElement("span");
        label.textContent = node.title || "(no name)";
        li.appendChild(label);

        li.dataset.id = node.id;
        li.draggable = true;

        li.addEventListener("click", () => loadBookmarks(node.id));

        folderList.appendChild(li);

        node.children.forEach(child => render(child, depth + 1));
      }
    }

    // tree[0] 自体（id: "0"）は描画せず、その子供だけを処理
    tree[0].children.forEach(child => render(child));

    // デフォルトでBookmark Barを読み込む（通常は id "1"）
    const bookmarkBar = tree[0].children?.find(node => node.id === "1");
    if (bookmarkBar) loadBookmarks(bookmarkBar.id);
  });
}

function loadBookmarks(folderId) {
  chrome.bookmarks.getChildren(folderId, (bookmarks) => {
    currentBookmarks = bookmarks;
    renderBookmarks(bookmarks);
  });
}

function renderBookmarks(bookmarks) {
  const bookmarkList = document.getElementById("bookmark-list");
  bookmarkList.innerHTML = "";
  bookmarkList.ondragover = (e) => e.preventDefault();
  bookmarks.forEach((bm) => {
    if (bm.url) {
      let li = document.createElement("li");

      let favicon = document.createElement("img");
      favicon.src = `https://www.google.com/s2/favicons?sz=16&domain_url=${encodeURIComponent(bm.url)}`;
      favicon.className = "favicon";

      let a = document.createElement("a");
      a.href = bm.url;
      a.textContent = bm.title;
      a.target = "_blank";

let openBtn = document.createElement("button");
openBtn.textContent = "🔗";
openBtn.className = "action";
openBtn.onclick = () => {
        chrome.tabs.create({
          url: bm.url,
          active: false, // ← バックグラウンドで開く
        });
      };

let copyBtn = document.createElement("button");
copyBtn.textContent = "📋";
copyBtn.className = "action";
copyBtn.title = "リンクをコピー";
copyBtn.onclick = () => {
  navigator.clipboard.writeText(bm.url).then(() => {
    copyBtn.textContent = "✅";  // 一瞬チェックマークでフィードバック
    setTimeout(() => (copyBtn.textContent = "📋"), 1000);
  });
};

let renameBtn = document.createElement("button");
renameBtn.textContent = "✏";
renameBtn.className = "action";

let deleteBtn = document.createElement("button");
deleteBtn.textContent = "🗑";
deleteBtn.className = "action";

// ボタン用のラッパーdiv
let actionWrapper = document.createElement("div");
actionWrapper.className = "action-wrapper";
actionWrapper.appendChild(openBtn);    // ← 追加
actionWrapper.appendChild(copyBtn);   // 📋 ← 新規追加
actionWrapper.appendChild(renameBtn);
actionWrapper.appendChild(deleteBtn);

// イベントはこのままでOK
renameBtn.onclick = () => {
  let newTitle = prompt("新しい名前を入力してください:", bm.title);
  if (newTitle) {
    chrome.bookmarks.update(bm.id, { title: newTitle }, () => loadBookmarks(bm.parentId));
  }
};

deleteBtn.onclick = () => {
  if (confirm("このブックマークを削除しますか？")) {
    chrome.bookmarks.remove(bm.id, () => loadBookmarks(bm.parentId));
  }
};

// append 順を調整
li.appendChild(favicon);
li.appendChild(a);
li.appendChild(actionWrapper); // ←ボタンを1つにまとめた要素を追加

            bookmarkList.appendChild(li);

      li.addEventListener("dragstart", (e) => {
        e.dataTransfer.setData("text/plain", bm.id);
      });

      li.addEventListener("drop", (e) => {
        e.preventDefault();
        const draggedId = e.dataTransfer.getData("text/plain");
        const targetId = bm.id;
        const draggedIndex = currentBookmarks.findIndex(b => b.id === draggedId);
        const targetIndex = currentBookmarks.findIndex(b => b.id === targetId);
        if (draggedIndex > -1 && targetIndex > -1 && draggedIndex !== targetIndex) {
          const moved = currentBookmarks.splice(draggedIndex, 1)[0];
          currentBookmarks.splice(targetIndex, 0, moved);
          const newOrder = currentBookmarks.map(b => b.id);
          for (let i = 0; i < newOrder.length - 1; i++) {
            chrome.bookmarks.move(newOrder[i], {parentId: bm.parentId, index: i});
          }
          loadBookmarks(bm.parentId);
        }
      });
    }
  });
}

document.addEventListener("DOMContentLoaded", () => {
  loadFolders();
  const searchBox = document.getElementById("search-box");
  if (searchBox) {
    searchBox.addEventListener("input", () => {
      const query = searchBox.value.toLowerCase();
      const filtered = currentBookmarks.filter(bm => bm.url && bm.title.toLowerCase().includes(query));
      renderBookmarks(filtered);
    });
  }
});